#pragma once

#include "Common/json/json.hpp"
#include "Common/FSecure/WinHttp/WebProxy.h"
#include "Common/FSecure/WinHttp/Constants.h"

using json = nlohmann::json; //for easy parsing of json API: https://github.com/nlohmann/json

namespace FSecure
{
	class Dropbox
	{
	public:

		/// Constructor for the Dropbox Api class.
		/// @param token - the token generated by Dropbox when an "app" is installed (either app or full dbx access)
		/// @param proxyString - the proxy to use
		Dropbox(std::string const& userAgent, std::string const& token, std::string const& channelName);

		/// Default constructor.
		Dropbox() = default;

		/// Write a message as the contents of a file and upload to Dropbox.
		/// @param direction - the name of the file to upload
		/// @param data - the text of the message
		/// @param filename - optional custom filename for uploaded file
		void WriteMessageToFile(std::string const& direction = "", ByteView data = {}, std::string const& providedFilename = "");

		/// Upload a file in its entirety to Dropbox.
		/// @param path - path to file for upload
		void UploadFile(std::string const& path);

		/// Delete channel folder and all files within
		void DeleteAllFiles();

		/// Set the channel (i.e. Dropbox folder) that this object uses for communications
		/// @param channelName - the channel name Id (not name), for example CGPMGFGSH.
		void SetChannel(std::string const& channelName);

		/// set the token for this object.
		/// @param token - the textual api token.
		void SetToken(std::string const& token);

		/// Will list the created folders in Dropbox and if already preset return the channel name. If not already created,
		/// creates a new folder on Dropbox.
		/// @param channelName - the actual name of the folder to create, such as "files".
		/// @return - the channel name of the new or already existing channel.
		std::string CreateChannel(std::string const& channelName);

		/// List all the folders in the workspace the object's token is tied to.
		/// @return - a map of {channelName -> channelId}
		std::map<std::string, std::string> ListChannels();

		/// Get all of the files representing messages by a direction. This is a C3 specific method, used by a server relay to get client messages and vice versa.
		/// @param direction - the direction to search for (eg. "S2C").
		/// @return - a map of timestamp and file id, where id allows replies to be read later
		std::map<std::string, std::string> GetMessagesByDirection(std::string const& direction);

		/// Download file by its path.
		/// @param filename - path of file.
		/// @return - string of file content
		FSecure::ByteVector ReadFile(std::string const& filename);

		/// Delete a file
		/// @param filename - the full path of the file on Dropbox.
		void DeleteFile(std::string const& filename);

	private:

		/// The channel (i.e. folder) through which messages are sent and received, will be sent when the object is created.
		std::string m_Channel;

		/// The Dropbox API token that allows the object access to the account. Needs to be manually created as described in documentation.
		std::string m_Token;

		/// Hold proxy settings
		WinHttp::WebProxy m_ProxyConfig;

		/// Send http request, uses preset token for authentication
		FSecure::ByteVector SendHttpRequest(std::string const& host, std::string const& header = "", std::optional<WinHttp::ContentType> contentType = {}, ByteView data = {});

		/// Send http request with json data, uses preset token for authentication
		json SendJsonRequest(std::string const& url, json const& data);

		/// The user agent header
		std::string m_UserAgent;

	};

}

